var feedCache = {
    validCacheDuration: 5*60*1000, // 5 minutes
    lastRequestDate: null,
    data: [],
    getdata: function (url, useproxy, cb) {
        var self = this;
        var now = new Date();

        // return cached data when still valid
        if(this.lastRequestDate && now-this.lastRequestDate < this.validCacheDuration) {
            return cb(self.data);
        }

        function rss2html(jqueryEl) {
            // try to get some sensible data from this xml-"html"-mess
            return jqueryEl.html()
                .replace("<![CDATA[", "")
                .replace("<!--[CDATA[<p-->", "<p>")
                .replace("]]>", "")
                .replace("]]&gt;", "")
                .replace(/<\/?font[^>]*>/gi, "");
        }

        // renew cache otherwise
        $.ajax({
            url: (useproxy) ? 'http://proxy.doohmedia.net/get.php?url=' + $.base64.encode(url) : url,
            type: 'GET',
            success: function(data){
                var result = [];
                $(data).find("item").each(function () {
                    var item = $(this);
                    var title = rss2html(item.find("title"));
                    var desc = rss2html(item.find("description"));

                    if (!title && !desc) return false;
                    if (!title && desc === "<p></p>") return false;

                    result.push({title: title, desc: desc});
                });

                if(result.length > 0) {
                    // only update when we got results
                    self.data = result;
                    self.lastRequestDate = new Date();
                }

                return cb(self.data);
            },
            error: function(err) {
                console.log("error while getting feed data: ", err);
                // Return cache. Cache may be empty or stale, but there is nothing else we can do.
                return cb(self.data);
            }
        });
    }
};

var ticker = {
    currentFeedData: [],
    currentFeedIndex: -1,
    config: null,
    viewportHeight: null,
    init: function (config) {
        var self = this;

        if(!config.url) return;
        self.config = config;

        self._setupStyling();

        // get data for first run
        feedCache.getdata(self.config.url, self.config.useproxy, function (data) {
            self.currentFeedData = data;

            self._setupNewTicker();

            // refresh feed data every minute
            window.setInterval(function() { self._updateFeedData(); }, 60*1000);
        });
    },
    _setupStyling: function() {
        this.viewportHeight = $('#viewport').height();

        if(this.config.font && this.config.font.file && this.config.font.format) {
            var fontFamily = this.config.font.name || "customfont";

            $('head').prepend('<style type="text/css">@font-face {font-family: "' + fontFamily + '"; src: url("' + this.config.font.file + '") format("' + this.config.font.format + '");}</style>');
            $('body').css('font-family', fontFamily);
        }

        if(this.config.background.color) {
            $('body').css('background-color',this.config.background.color);
        }

        if(this.config.background.image) {
            $('body').css('background','url("' + this.config.background.image + '") no-repeat center fixed')
                .css('background-size', 'cover');
        }

        this._styleFonts();
    },
    _styleFonts: function () {
        if(this.config.text.size) $("p").css('font-size', this.config.text.size);
        if(this.config.text.color) $("p").css('color', this.config.text.color);
        if(this.config.text.align) $("p").css('text-align', this.config.text.align);
        if(this.config.headline.size) $("h1").css('font-size', this.config.headline.size);
        if(this.config.headline.color) $("h1").css('color', this.config.headline.color);
        if(this.config.spacer.color) $(".spacer").css('color', this.config.spacer.color);
        if(this.config.spacer.size) $(".spacer").css('font-size', this.config.spacer.size);
    },
    _updateFeedData: function () {
        var self = this;

        feedCache.getdata(self.config.url, self.config.useproxy, function (data) {
            self.currentFeedData = data;
        });
    },
    _setupNewTicker: function () {
        var self = this;

        // We can do nothing if we have no data yet.
        if (self.currentFeedData.length === 0) return;

        // Fill screen with items
        self._createTicker();
        setTimeout(function () {
            // Wait 500ms for second ticker to let the DOM-setup of the first one complete.
            self._createTicker();
        }, 500);

    },
    _createTicker: function () {
        var self = this;

        // We can do nothing if we have no data yet
        if (this.currentFeedData.length === 0) return;

        // Get the top position of the last ticker item (or the bottom position of the viewport for the first ticker)
        var lastTicker = $(".tickercontainer:last").get(0);
        var top = (lastTicker) ? $(lastTicker).offset().top + $(lastTicker).height() : this.viewportHeight;

        // Add ticker container to DOM
        var newContainer = $('<div class="tickercontainer"></div>');
        $("#viewport").append(newContainer);

        // Add ticker items to the container
        for(var i=0;i<this.currentFeedData.length;i++) {
            if(this.config.spacer.text.length > 0) {
                var newItem = $('<div class="tickeritem"><h1>' + this.currentFeedData[i].title + '</h1>' + this.currentFeedData[i].desc + '</div><div class="spacer">' + this.config.spacer.text + '</div>');
            } else {
                var newItem = $('<div class="tickeritem"><h1>' + this.currentFeedData[i].title + '</h1>' + this.currentFeedData[i].desc + '</div>');
            }

            $(newItem).children("p").addClass("text");
            $(newContainer).append(newItem);
        }

        self._styleFonts();

        // Set top position
        $(newContainer).css('top', top + "px");

        // Calculate scrolling position and speed according to set scrolling speed.
        var distance = top + $(newContainer).height() + 20; // travel 20px out of screen to give some space for the fonts descent line
        var time = Math.round((distance / 100) * this.config.speed);

        $(newContainer).css('transition', 'transform ' + time + 's linear');
        $(newContainer).css('transform', 'translate3d(0px, -' + distance + 'px ,0px)');

        // Create a new ticker and clean up this one after it has scrolled out of screen.
        $(newContainer).on( 'transitionend', function() {
            self._createTicker();
            $(newContainer).remove();
        });
    }
};